#! /usr/bin/env python

"""
Experiment to run NBS-algorithm over all optimal domains.
"""

import os
import os.path
import platform

from lab.environments import LocalEnvironment, BaselSlurmEnvironment

from downward.experiment import FastDownwardExperiment
from downward import suites
from downward.reports.absolute import AbsoluteReport
from downward.reports.summary import SummaryReport


# Suites
SUITE_OPTIMAL = [ # needs: BENCHMARKS_DIR
    'blocks', 'depot', 'driverlog', 'elevators-opt08-strips', 'elevators-opt11-strips',
    'floortile-opt11-strips', 'ged-opt14-strips', 'gripper', 'hiking-opt14-strips', 
    'logistics00', 'miconic', 'nomystery-opt11-strips', 'openstacks-opt08-strips', 
    'openstacks-strips', 'pegsol-08-strips', 'pegsol-opt11-strips', 'psr-small', 'rovers',
    'satellite', 'scanalyzer-08-strips', 'scanalyzer-opt11-strips', 'storage', 
    'termes-opt18-strips','tpp', 'transport-opt08-strips', 'transport-opt11-strips', 
    'transport-opt14-strips', 'trucks-strips', 'zenotravel']
SUITE_SHORT = [ # needs: BENCHMARK_DIR
    'driverlog:p01.pddl'
]
SUITE_ALL_OPTIMAL = [ 
    'blocks', 'depot', 
    'driverlog', 'elevators-opt08-strips', 'elevators-opt11-strips', 'floortile-opt11-strips', 
    'floortile-opt14-strips', 'ged-opt14-strips','gripper', 
    'hiking-opt14-strips', 'logistics00', 'maintenance-opt14-adl', 'miconic', 
    'nomystery-opt11-strips', 'openstacks-opt08-strips', 'openstacks-opt11-strips',
    'pegsol-08-strips', 'pegsol-opt11-strips', 'petri-net-alignment-opt18-strips',  
    'psr-small', 'rovers', 'satellite', 'scanalyzer-08-strips', 'scanalyzer-opt11-strips', 
    'storage', 'termes-opt18-strips', 'tpp', 'transport-opt08-strips', 'transport-opt11-strips', 
    'transport-opt14-strips',  'trucks-strips', 'visitall-opt11-strips', 'visitall-opt14-strips', 
    'zenotravel'
]

SUITE_NONE_SOLVED = [
    'agricola-opt18-strips', 'childsnack-opt14-strips', 
]

SUITE_ONLY_ASTAR = [
    'airport', 'barman-opt11-strips', 'barman-opt14-strips', 'data-network-opt18-strips', 'grid',
    'openstacks-opt14-strips', 'organic-synthesis-split-opt18-strips', 'parking-opt11-strips', 
    'parking-opt14-strips', 'pathways', 'pathways-noneg', 'pipesworld-notankage', 'schedule', 
    'snake-opt18-strips', 'sokoban-opt08-strips', 'sokoban-opt11-strips', 'spider-opt18-strips', 
    'tetris-opt14-strips', 'tidybot-opt11-strips', 'tidybot-opt14-strips', 
    'woodworking-opt11-strips', 
]

SUITE_ONLY_FEW_SOLVED = [
    'freecell', 'logistics98', 'movie', 'mprime', 'mystery', 'openstacks-strips', 
    'organic-synthesis-opt18-strips', 'parcprinter-08-strips', 'parcprinter-opt11-strips', 
    'pipesworld-tankage', 'woodworking-opt08-strips', 
]

SUITE_AXIOMS = [
    'assembly', 'miconic-fulladl', 'openstacks', 'openstacks-opt08-adl', 'optical-telegraphs', 
    'philosophers', 'psr-large', 'psr-middle', 'trucks'
]

SUITE_CONDITIONAL_EFFECTS = [
    'caldera-opt18-adl', 'caldera-split-opt18-adl', 'cavediving-14-adl', 'citycar-opt14-adl', 
    'miconic-simpleadl', 'nurikabe-opt18-adl', 'settlers-opt18-adl', 
]



NODE = platform.node()
REMOTE = NODE.endswith(".scicore.unibas.ch") or NODE.endswith(".cluster.bc2.ch")
BENCHMARKS_DIR = os.environ["DOWNWARD_BENCHMARKS"]
BENCHMARK_DIR = os.environ["MASTER_BENCHMARK"]
REPO = os.environ['DOWNWARD_REPO']
REV = 'tip' # Branch here: defaut, tip, specific_revision, tag.
if REMOTE:
    ENV = BaselSlurmEnvironment(email="marvin.buff@unibas.ch")
    SUITE = SUITE_ALL_OPTIMAL
    DIR = BENCHMARKS_DIR
else:
    ENV = LocalEnvironment(processes=4)
    SUITE = SUITE_SHORT
    DIR = BENCHMARK_DIR

ATTRIBUTES_STANDARD = [
    'error', 'expanded', 'evaluated', 'cost', 'memory', # Standard 1
    'plan_length', 'planner_time', 'search_time', # Standard 2
    'ratio_fb_actions', 'ratio_fb_branching', 'meeting_point', # Combined forward/backward
    'ratio_jump_expanded', 'ratio_jump_evaluated', # after last jump
    'ratio_jump_generated', 'jump_expanded'] 
ATTRIBUTES_DIRECTIONAL = [ # forward/ backward 
    'f_expanded', 'b_expanded', 
    'f_evaluated', 'b_evaluated', 
    'f_generated', 'b_generated']
ATTRIBUTES_BACKWARD = [ # specific backward
    'b_pruned', 'b_initial_goals']
ATTIRBUTES_EXTENDED = [ # unused but implemented
    'dead_ends', 'reopened', # Standard
    'frontier_meetings' ,'f_ex_before_meeting', 'b_ex_before_meeting' # Meetings
]

ATTRIBUTES = ATTRIBUTES_STANDARD + ATTRIBUTES_DIRECTIONAL + ATTRIBUTES_BACKWARD


# Create a new experiment.
exp = FastDownwardExperiment(environment=ENV)
# Add custom parser.
exp.add_parser(exp.EXITCODE_PARSER)
exp.add_parser(exp.TRANSLATOR_PARSER)
exp.add_parser(exp.SINGLE_SEARCH_PARSER)
exp.add_parser(exp.PLANNER_PARSER)
exp.add_parser('parsers/simple-parser.py')

DRIVER_OPTIONS = ['--overall-time-limit', '30m', '--overall-memory-limit', '3584M']

exp.add_suite(DIR, SUITE)
exp.add_algorithm(
    'astar_blind', REPO, REV, ['--search', 'astar(blind)'], 
    driver_options=DRIVER_OPTIONS)
exp.add_algorithm(
    'astar_max', REPO, REV, ['--search', 'astar(hmax)'], 
    driver_options=DRIVER_OPTIONS)
exp.add_algorithm(
    'astar_lmcut', REPO, REV, ['--search', 'astar(lmcut)'], 
    driver_options=DRIVER_OPTIONS)

exp.add_algorithm(
    'nbs_blind', REPO, REV, ['--search', 'nbs(eval=blind, b_eval=blind(transform=backward))'], 
    driver_options=DRIVER_OPTIONS)
exp.add_algorithm(
    'nbs_max', REPO, REV, ['--search', 'nbs(eval=hmax, b_eval=hmax(transform=backward))'], 
    driver_options=DRIVER_OPTIONS)
exp.add_algorithm(
    'nbs_lmcut', REPO, REV, ['--search', 'nbs(eval=lmcut, b_eval=lmcut(transform=backward))'], 
    driver_options=DRIVER_OPTIONS)

##Add step that writes experiment files to disk.
exp.add_step('build', exp.build)

# Add step that executes all runs.
exp.add_step('start', exp.start_runs)

# Add step that collects properties from run directories and
# writes them to *-eval/properties.
exp.add_fetcher(name='fetch')

# Make a report.
exp.add_report(
    AbsoluteReport(attributes=ATTRIBUTES),
    outfile='report.html')

exp.add_report(
    SummaryReport(attributes=ATTRIBUTES, filter_algorithm=['astar_blind','nbs_blind']),
    outfile='summary_blind.html')
exp.add_report(
    SummaryReport(attributes=ATTRIBUTES, filter_algorithm=['astar_hm','nbs_hm']),
    outfile='summary_hm.html')
exp.add_report(
    SummaryReport(attributes=ATTRIBUTES, filter_algorithm=['astar_lmcut','nbs_lmcut']),
    outfile='summary_lmcut.html')
exp.add_report(
    SummaryReport(attributes=ATTRIBUTES, filter_algorithm=['astar_max','nbs_max']),
    outfile='summary_max.html')

# Parse the commandline and run the specified steps.
exp.run_steps()
